"use client";
import Link from "next/link";
import { useRouter } from "next/router";
import { useSession, signOut } from "next-auth/react";
import { useSidebar } from "@/contexts/SidebarContext";
import { useLanguage } from "@/contexts/LanguageContext";
import LanguageSwitcher from "@/components/LanguageSwitcher";
import { Ticket, LogOut, User } from "lucide-react";

export default function Navbar() {
  const { data: session, status } = useSession();
  const router = useRouter();
  const { isCollapsed } = useSidebar();
  const { t } = useLanguage();

  const isActive = (path: string) =>
    router.pathname === path || router.pathname.startsWith(path + "/");

  const handleSignOut = () => signOut({ callbackUrl: "/login" });

  return (
    <nav
      className={`sticky top-0 z-40 bg-sidebar border-b border-border text-foreground transition-all duration-300 ${
        isCollapsed ? "lg:ml-16" : "lg:ml-64"
      }`}
    >
      <div className='w-full px-4 lg:px-6'>
        <div className='flex items-center justify-between h-16'>
          {/* Logo and Hospital Name (Mobile Only) */}
          <Link
            href='/'
            className='lg:hidden flex items-center gap-2 hover:opacity-80 transition cursor-pointer'
          >
            <div className='w-8 h-8 bg-primary-500 rounded-lg flex items-center justify-center flex-shrink-0 hover:bg-primary-600 transition'>
              <Ticket className='w-5 h-5 text-primary-fore' />
            </div>
            <span className='text-base font-bold text-foreground'>
              {t("nav.hospitalPortal")}
            </span>
          </Link>

          {/* Empty space on desktop */}
          <div className='hidden lg:block'></div>

          {/* User Section */}
          <div className='flex items-center gap-2 sm:gap-3'>
            {/* Language Switcher */}
            <LanguageSwitcher />

            {status === "authenticated" && session?.user ? (
              <>
                <Link
                  href='/profile'
                  className={`flex items-center gap-1 sm:gap-2 px-2 sm:px-3 py-2 rounded-md text-sm font-medium ${
                    isActive("/profile")
                      ? "bg-sidebar-accent text-foreground"
                      : "text-foreground/80 hover:text-foreground hover:bg-sidebar-accent"
                  }`}
                >
                  <div className='w-6 h-6 bg-sidebar-accent rounded-full flex items-center justify-center flex-shrink-0'>
                    <User className='w-4 h-4 text-foreground/80' />
                  </div>
                  <span className='max-w-36 truncate hidden sm:inline'>
                    {session.user.email}
                  </span>
                </Link>

                <button
                  onClick={handleSignOut}
                  className='flex items-center gap-1 sm:gap-2 px-2 sm:px-3 py-2 rounded-md text-sm font-medium text-foreground/80 hover:text-foreground hover:bg-sidebar-accent'
                >
                  <LogOut className='w-4 h-4' />
                  <span className='hidden sm:inline'>{t("nav.signOut")}</span>
                </button>
              </>
            ) : (
              <Link
                href='/login'
                className='px-3 sm:px-4 py-2 rounded-md text-sm font-medium bg-sidebar-accent text-foreground hover:opacity-90'
              >
                {t("nav.signIn")}
              </Link>
            )}
          </div>
        </div>
      </div>
    </nav>
  );
}
